// Helpers
const $ = (sel, root = document) => root.querySelector(sel);
const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

function showToast(msg) {
  const toast = $("#toast");
  const toastText = $("#toastText");
  toastText.textContent = msg;
  toast.hidden = false;
  toast.style.opacity = "0";
  toast.style.transform = "translateY(6px)";
  requestAnimationFrame(() => {
    toast.style.transition = "opacity .18s ease, transform .18s ease";
    toast.style.opacity = "1";
    toast.style.transform = "translateY(0)";
  });
  window.clearTimeout(showToast._t);
  showToast._t = window.setTimeout(() => {
    toast.style.opacity = "0";
    toast.style.transform = "translateY(6px)";
    window.setTimeout(() => (toast.hidden = true), 220);
  }, 2400);
}

// Mobile nav
const navToggle = $("#navToggle");
const navMenu = $("#navMenu");

navToggle?.addEventListener("click", () => {
  const open = navMenu.classList.toggle("is-open");
  navToggle.setAttribute("aria-expanded", String(open));
});

$$('#navMenu a').forEach(a => {
  a.addEventListener("click", () => {
    navMenu.classList.remove("is-open");
    navToggle.setAttribute("aria-expanded", "false");
  });
});

// Year
$("#year").textContent = String(new Date().getFullYear());

// Back to top
$("#backToTop")?.addEventListener("click", () => {
  window.scrollTo({ top: 0, behavior: "smooth" });
});

// Modal (Rules)
const rulesModal = $("#rulesModal");
const openRules = $("#openRules");
const closeRules = $("#closeRules");

function openModal() {
  rulesModal.setAttribute("aria-hidden", "false");
  document.body.style.overflow = "hidden";
}
function closeModal() {
  rulesModal.setAttribute("aria-hidden", "true");
  document.body.style.overflow = "";
}
openRules?.addEventListener("click", openModal);
closeRules?.addEventListener("click", closeModal);
rulesModal?.addEventListener("click", (e) => {
  const t = e.target;
  if (t && (t.dataset.close === "true" || t.classList.contains("modal__overlay"))) closeModal();
});
document.addEventListener("keydown", (e) => {
  if (e.key === "Escape" && rulesModal?.getAttribute("aria-hidden") === "false") closeModal();
});

// FAQ accordion
const faqItems = $$(".faq__item");
faqItems.forEach((btn) => {
  const panel = btn.nextElementSibling;
  btn.addEventListener("click", () => {
    const expanded = btn.getAttribute("aria-expanded") === "true";
    // Close others
    faqItems.forEach((b) => {
      if (b !== btn) {
        b.setAttribute("aria-expanded", "false");
        const p = b.nextElementSibling;
        if (p) p.style.display = "none";
      }
    });
    btn.setAttribute("aria-expanded", String(!expanded));
    if (panel) panel.style.display = expanded ? "none" : "block";
  });
});

// Animated counters (simple)
function animateCounters() {
  const els = $$("[data-counter]");
  const reduceMotion = window.matchMedia("(prefers-reduced-motion: reduce)").matches;

  const run = (el) => {
    const target = Number(el.dataset.counter || "0");
    const duration = reduceMotion ? 1 : 900;
    const start = performance.now();
    const from = 0;

    const tick = (t) => {
      const p = Math.min(1, (t - start) / duration);
      const eased = 1 - Math.pow(1 - p, 3);
      const value = Math.round(from + (target - from) * eased);
      el.textContent = value.toLocaleString("pt-BR");
      if (p < 1) requestAnimationFrame(tick);
    };
    requestAnimationFrame(tick);
  };

  // trigger once when visible
  const io = new IntersectionObserver((entries) => {
    for (const entry of entries) {
      if (entry.isIntersecting) {
        run(entry.target);
        io.unobserve(entry.target);
      }
    }
  }, { threshold: 0.6 });

  els.forEach((el) => io.observe(el));
}
animateCounters();

// Countdown to next Sunday 20:00 (America/Sao_Paulo local time)
// JS Date uses local timezone of the device; that's OK for a landing page.
// If you host for global users, you can adapt using Intl time zones.
function nextSundayAt20() {
  const now = new Date();
  const d = new Date(now);
  const day = d.getDay(); // 0=Sunday
  const diff = (7 - day) % 7; // days until Sunday
  d.setDate(d.getDate() + diff);
  d.setHours(20, 0, 0, 0);

  // If it's already Sunday past 20:00, go next week
  if (day === 0 && now.getTime() >= d.getTime()) {
    d.setDate(d.getDate() + 7);
  }
  return d;
}

let target = nextSundayAt20();

function pad2(n) { return String(n).padStart(2, "0"); }

function updateCountdown() {
  const now = new Date();
  let ms = target.getTime() - now.getTime();

  if (ms <= 0) {
    // reset to next draw
    target = nextSundayAt20();
    ms = target.getTime() - now.getTime();
    showToast("Sorteio rolando! Próximo agendado automaticamente.");
  }

  const total = Math.floor(ms / 1000);
  const days = Math.floor(total / 86400);
  const hours = Math.floor((total % 86400) / 3600);
  const mins = Math.floor((total % 3600) / 60);
  const secs = total % 60;

  $("#d").textContent = pad2(days);
  $("#h").textContent = pad2(hours);
  $("#m").textContent = pad2(mins);
  $("#s").textContent = pad2(secs);

  // progress bar: from last Sunday 20:00 to next Sunday 20:00
  const end = target.getTime();
  const start = new Date(target);
  start.setDate(start.getDate() - 7);
  const span = end - start.getTime();
  const elapsed = now.getTime() - start.getTime();
  const pct = Math.max(0, Math.min(100, (elapsed / span) * 100));
  $("#progressBar").style.width = `${pct.toFixed(1)}%`;
  $("#progressText").textContent = `${pct.toFixed(0)}% até o sorteio`;
}
updateCountdown();
setInterval(updateCountdown, 1000);

// Plans buttons
$$("[data-plan]").forEach((btn) => {
  btn.addEventListener("click", () => {
    const plan = btn.getAttribute("data-plan");
    showToast(`Inscrição iniciada: ${plan} (demo)`);
    document.querySelector("#contact")?.scrollIntoView({ behavior: "smooth" });
    $("#topic").value = "Inscrição / dúvidas";
    $("#message").value = `Olá! Quero participar do ${plan}. Como faço o check-in e horários?`;
  });
});

// Copy invite (replace with your real invite link)
const INVITE_LINK = "https://discord.gg/SEUCONVITE";
$("#copyInvite")?.addEventListener("click", async () => {
  try {
    await navigator.clipboard.writeText(INVITE_LINK);
    showToast("Convite copiado!");
  } catch {
    showToast("Não consegui copiar. Copie manualmente: " + INVITE_LINK);
  }
});

// Contact form validation + fake submit
const form = $("#contactForm");
function setFieldError(name, msg) {
  const el = document.querySelector(`[data-error-for="${name}"]`);
  if (el) el.textContent = msg || "";
}
function clearErrors() {
  ["name","discord","topic","message"].forEach((n) => setFieldError(n, ""));
}

form?.addEventListener("submit", (e) => {
  e.preventDefault();
  clearErrors();

  const data = new FormData(form);
  const name = String(data.get("name") || "").trim();
  const discord = String(data.get("discord") || "").trim();
  const topic = String(data.get("topic") || "").trim();
  const message = String(data.get("message") || "").trim();

  let ok = true;
  if (name.length < 2) { setFieldError("name", "Informe seu nome."); ok = false; }
  if (discord.length < 3) { setFieldError("discord", "Informe seu usuário no Discord."); ok = false; }
  if (!topic) { setFieldError("topic", "Selecione um assunto."); ok = false; }
  if (message.length < 10) { setFieldError("message", "Escreva uma mensagem com mais detalhes."); ok = false; }

  if (!ok) {
    showToast("Revise os campos destacados.");
    return;
  }

  // Demo submission
  showToast("Mensagem enviada! (demo)");
  $("#formHint").textContent = `Recebido, ${name}! Vamos te chamar no Discord: ${discord}`;
  form.reset();
});

// Event date label (cosmetic)
(function setEventLabel(){
  // Example: Saturday 20:00 (next Saturday)
  const now = new Date();
  const d = new Date(now);
  const day = d.getDay(); // 0 Sunday
  // days until Saturday (6)
  const diff = (6 - day + 7) % 7;
  d.setDate(d.getDate() + diff);
  d.setHours(20,0,0,0);

  const weekday = new Intl.DateTimeFormat("pt-BR", { weekday: "long" }).format(d);
  const date = new Intl.DateTimeFormat("pt-BR", { day: "2-digit", month: "2-digit" }).format(d);
  $("#eventDate").textContent = `${weekday} • ${date} • 20:00`;
})();


// =========================================================
// MOBILE CTA FIXO (sem mexer no HTML original)
// =========================================================
(function mobileCta() {
  const cta = document.createElement("div");
  cta.className = "mobile-cta";
  cta.innerHTML = `
    <a class="btn btn--primary" href="https://discord.com/" target="_blank" rel="noreferrer">
      Entrar no Discord <span class="btn__icon" aria-hidden="true">➜</span>
    </a>
    <button class="btn btn--secondary" id="mCtaTournaments">
      Ver torneios
    </button>
  `;
  document.body.appendChild(cta);

  const btn = cta.querySelector("#mCtaTournaments");
  btn?.addEventListener("click", () => {
    document.querySelector("#tournaments")?.scrollIntoView({ behavior: "smooth" });
  });

  // esconde CTA quando usuário está no topo da página (opcional)
  function toggleCtaVisibility() {
    const y = window.scrollY || 0;
    cta.style.opacity = y < 80 ? "0" : "1";
    cta.style.pointerEvents = y < 80 ? "none" : "auto";
  }
  toggleCtaVisibility();
  window.addEventListener("scroll", toggleCtaVisibility, { passive: true });
})();

// Fecha o menu quando clicar fora (mobile)
(function closeNavOnOutsideClick(){
  const toggle = document.querySelector("#navToggle");
  const menu = document.querySelector("#navMenu");
  if (!toggle || !menu) return;

  document.addEventListener("click", (e) => {
    const isOpen = menu.classList.contains("is-open");
    if (!isOpen) return;

    const clickedInside = menu.contains(e.target) || toggle.contains(e.target);
    if (!clickedInside) {
      menu.classList.remove("is-open");
      toggle.setAttribute("aria-expanded", "false");
    }
  });
})();
